<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Classes;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ClassController extends Controller
{
    public function index(Request $request){
        return view('class.index');

    }

    public function create_class (Request $request){
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'room_id' => 'required|integer|exists:rooms,id', // Assuming room_id exists in the rooms table
            'start_date' => 'required|date_format:Y-m-d', // Validates date in YYYY-MM-DD format
            'start_time' => 'required|date_format:H:i:s', // Validates time in HH:mm:ss format
            'end_date' => 'required|date_format:Y-m-d|after_or_equal:start_date',
            'end_time' => 'required|date_format:H:i:s|after:start_time', // Ensures end time is after start time
            'group_id' => 'required|integer|exists:groups,id', // Assuming group_id exists in the groups table
            'repeat_type' => 'required|in:Once,Weekly,Monthly', // Adjust options according to your needs
            'repeat_until' => 'nullable|date_format:Y-m-d|after_or_equal:start_date',
        ]);


        $startDate = Carbon::createFromFormat('Y-m-d', $request->start_date);
        $repeatType = $request->repeat_type;
        $repeatUntil = Carbon::createFromFormat('Y-m-d', $request->repeat_until);

        $dates = [];
        $currentDate = $startDate;

        while ($currentDate->lte($repeatUntil)) {
            $dates[] = $currentDate->toDateString();
            if ($repeatType == 'Weekly') {
                $currentDate->addWeek();
            } elseif ($repeatType == 'Monthly') {
                $currentDate->addMonth();
            } else {
                break;
            }
        }

        // Loop through each date to check for class clashes
        foreach ($dates as $date) {
            $startDateTime = Carbon::parse($date . ' ' . $request->start_time);
            $endDateTime = Carbon::parse($date . ' ' . $request->end_time);
            $existingClass = Classes::where('room_id', $request->room_id)
                ->where(function ($query) use ($startDateTime, $endDateTime) {
                    $query->where(function ($q) use ($startDateTime, $endDateTime) {
                        $q->where('start_date_time', '<', $endDateTime)
                            ->where('end_date_time', '>', $startDateTime);
                    });
                })
                ->exists();
            if ($existingClass) {
                return redirect()->back()->withErrors(['error' => "Class clashes with another class scheduled on {$date}. Please choose a different time."]);
            }

            $class = new Classes();
            $class->title = $request->title;
            $class->description = $request->description;
            $class->room_id = $request->room_id;
            $class->start_date_time = $startDateTime;
            $class->end_date_time = $endDateTime;
            $class->group_id = $request->group_id;
            $class->repeat_type = $request->repeat_type;
            $class->status = 'Approved';
            $class->requested_by_id = Auth::user()->id;
            
            $class->save();
        }
        return redirect()->back()->with('success', 'Class created successfully.');
    }

    public function update_class_by_id (Request $request, $id){
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'room_id' => 'required|integer|exists:rooms,id', // Assuming room_id exists in the rooms table
            'start_date' => 'required|date_format:Y-m-d', // Validates date in YYYY-MM-DD format
            'start_time' => 'required|date_format:H:i:s', // Validates time in HH:mm:ss format
            'end_date' => 'required|date_format:Y-m-d|after_or_equal:start_date',
            'end_time' => 'required|date_format:H:i:s|after:start_time', // Ensures end time is after start time
            'group_id' => 'required|integer|exists:groups,id', // Assuming group_id exists in the groups table
        ]);

        $startDateTime = Carbon::parse($request->start_date . ' ' . $request->start_time);
        $endDateTime = Carbon::parse($request->end_date . ' ' . $request->end_time);
        $existingClass = Classes::where('room_id', $request->room_id)
            ->where('id', '!=', $id)
            ->where(function ($query) use ($startDateTime, $endDateTime) {
                $query->where(function ($q) use ($startDateTime, $endDateTime) {
                    $q->where('start_date_time', '<', $endDateTime)
                        ->where('end_date_time', '>', $startDateTime);
                });
            })
            ->exists();
        if ($existingClass) {
            return redirect()->back()->withErrors(['error' => "Class clashes with another class scheduled on {$request->start_date}. Please choose a different time."]);
        }


        $class = Classes::findOrFail($id);
        $class->title = $request->title;
        $class->description = $request->description;
        $class->room_id = $request->room_id;
        $class->start_date_time = $request->start_date . ' ' . $request->start_time;
        $class->end_date_time = $request->end_date . ' ' . $request->end_time;
        $class->group_id = $request->group_id;
        // dd($request);
        $class->save();

        return redirect()->back()->with('success', 'Class updated successfully.')->with('updated_class_id', $id);
    }

    public function recreate_class_by_id (Request $request, $id){
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'room_id' => 'required|integer|exists:rooms,id', // Assuming room_id exists in the rooms table
            'start_date' => 'required|date_format:Y-m-d', // Validates date in YYYY-MM-DD format
            'start_time' => 'required|date_format:H:i:s', // Validates time in HH:mm:ss format
            'end_date' => 'required|date_format:Y-m-d|after_or_equal:start_date',
            'end_time' => 'required|date_format:H:i:s|after:start_time', // Ensures end time is after start time
            'group_id' => 'required|integer|exists:groups,id', // Assuming group_id exists in the groups table
            'repeat_type' => 'required|in:Once,Weekly,Monthly', // Adjust options according to your needs
        ]);
        $class = new Classes();
        $class->title = $request->title;
        $class->description = $request->description;
        $class->room_id = $request->room_id;
        $class->start_date_time = $request->start_date . ' ' . $request->start_time;
        $class->end_date_time = $request->end_date . ' ' . $request->end_time;
        $class->group_id = $request->group_id;
        $class->repeat_type = $request->repeat_type;
        $class->status = 'Approved';
        $class->save();

        return redirect()->back()->with('success', 'Class recreated successfully.')->with('updated_class_id', $id);
    }

    public function approve_class_by_id(Request $request, $id) {
        $request->validate([
            'remark' => 'nullable|string|max:1000',
        ]);
        $class = Classes::findOrFail($id);
        $class->approval_remark = $request->approval_remark;
        $class->approval_date_time = now();
        $class->approved_by_id = Auth::user()->id;
        $class->save();

        return redirect()->back()->with('success', 'Class approved successfully.');

    }

    public function delete_class_by_id ($id) {
        $class = Classes::findOrFail($id);
        $class->delete();

        return redirect()->back()->with('success', 'Class deleted successfully.');
    }
}
