<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;

class UserController extends Controller
{
    private function get_options($body, $uri) {
        $appKey = env('HIKCENTRAL_APP_KEY');
        $appSecret = env('HIKCENTRAL_APP_SECRET');
        $date = Carbon::now('GMT')->format('D, d M Y H:i:s \G\M\T');
        $method = 'POST';
        $contentType = 'application/json';

        // Generate Content-MD5
        $contentMd5 = base64_encode(md5($body, true));

        // Create the string to sign
        $stringToSign = "{$method}\n" .
                        "*/*\n" .
                        "{$contentMd5}\n" .
                        "{$contentType}\n" .
                        "{$date}\n" .
                        "x-ca-key:{$appKey}\n" .
                        "x-ca-timestamp:" . now()->timestamp * 1000 . "\n" .
                        $uri;

        // Generate the HmacSHA256 signature
        $signature = base64_encode(hash_hmac('sha256', $stringToSign, $appSecret, true));

        $options = [
            'headers' => [
                'Accept' => '*/*',
                'Content-MD5' => $contentMd5,
                'Content-Type' => $contentType,
                'Date' => $date,
                'X-Ca-Key' => $appKey,
                'X-Ca-Signature' => $signature,
                'X-Ca-Timestamp' => now()->timestamp * 1000,
                'X-Ca-Signature-Headers' => 'x-ca-key,x-ca-timestamp',
            ],
            'body' => $body,
        ];

        return $options;

    }

    public function index(Request $request){
        return view('user.index');
    }

    public function create_user (Request $request){
        $request->validate([
            'staff_id' => 'required|alpha_num|max:20|unique:users,staff_id', // Staff ID should be unique, alphanumeric and max 20 chars
            'email' => 'required|email|unique:users,email', // Email should be unique and a valid email format
            'first_name' => 'required|string|max:50', // First name should be string and max 50 chars
            'last_name' => 'required|string|max:50', // Last name should be string and max 50 chars
            'full_name' => 'required|string|max:100', // Full name should be string and max 100 chars
            'phone_no' => 'nullable|digits_between:8,15', // Phone number can be null, but should be between 10 and 15 digits
            'address' => 'nullable|string|max:255', // Address can be null, but max 255 chars
            'nric' => 'required|string|size:12|unique:users,nric', // NRIC should be exactly 12 chars and unique
            'remark' => 'nullable|string|max:500', // Remark can be null, max 500 chars
            'gender' => 'required|in:1,0', // Gender must be male, female, or other
            'type' => 'required|in:Administrator,Management,Lecturer,Student Lead,Student', // Type must be admin, user, or manager
            'password' => 'required|min:8|confirmed', // Password must be at least 8 chars and must match password_confirmation
        ]);
        $user = new User();
        $user->staff_id = $request->staff_id;
        $user->email = $request->email;
        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->full_name = $request->full_name;
        $user->phone_no = $request->phone_no;
        $user->address = $request->address;
        $user->nric = $request->nric;
        $user->remark = $request->remark;
        $user->gender = $request->gender;
        $user->type = $request->type;
        $user->password = Hash::make($request->password);
        if ($request->file('image')) {
            $image_file = $request->file('image');
            $filename = 'profile_' . time() . '.jpg';
            $image_file_path = '/' . $image_file->storeAs('/user/' . strtoupper($request->full_name) . '/profile_picture', $filename, 'public');
            $user->image_path = $image_file_path;
        }

        $client = new \GuzzleHttp\Client([
            'verify' => false, // Disable SSL verification
        ]);
        $uri = '/artemis/api/resource/v1/person/single/add';
        $body = json_encode(array_filter([
            "personCode"=> $request->staff_id,
            "personGivenName"=> $request->first_name,
            "personFamilyName"=> $request->last_name,
            "gender"=> $request->gender == 1 ? '1' : '2',
            "orgIndexCode"=> '1',
            "phoneNo"=> $request->phone_no,
            "email"=> $request->email,
            // "faces"=> [
            //     [
            //         "faceData" => $request->input('face')
            //     ]
            // ],
            "remark"=> $request->remark,
        ], function($value) {
            return !is_null($value);
        }));
        $options = $this->get_options($body, $uri);
        try {
            $response = $client->post(env('HIKCENTRAL_API_BASE_URL') . $uri, $options);
            $data = json_decode($response->getBody(), true);
            $user->hcp_id = $data['data'];
            $user->save();
            return redirect()->back()->with('success', 'User created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Cannot add user to Hik Central Professional.']);
        }
    }

    public function create_user_excel () {
        dd();
        return view('user.create-user');
    }

    public function update_user_by_id (Request $request, $id){
        $request->validate([
            'first_name' => 'required|string|max:50', // First name should be string and max 50 chars
            'last_name' => 'required|string|max:50', // Last name should be string and max 50 chars
            'full_name' => 'required|string|max:100', // Full name should be string and max 100 chars
            'phone_no' => 'nullable|digits_between:8,15', // Phone number can be null, but should be between 10 and 15 digits
            'address' => 'nullable|string|max:255', // Address can be null, but max 255 chars
            'remark' => 'nullable|string|max:500', // Remark can be null, max 500 chars
            'gender' => 'required|in:1,0', // Gender must be male, female, or other
            'type' => 'required|in:Administrator,Management,Lecturer,Student Lead,Student', // Type must be admin, user, or manager
        ]);
        $user = User::findOrFail($id);
        if ($user->staff_id == $request->staff_id) {}
        else {$request->validate(['staff_id' => 'required|alpha_num|max:20|unique:users,staff_id',]);}
        if ($user->email == $request->email) {}
        else {$request->validate(['email' => 'required|email|unique:users,email',]);}
        if ($user->nric == $request->nric) {}
        else {$request->validate(['nric' => 'required|string|size:12|unique:users,nric',]);}
        $user->staff_id = $request->staff_id;
        $user->email = $request->email;
        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->full_name = $request->full_name;
        $user->phone_no = $request->phone_no;
        $user->address = $request->address;
        $user->nric = $request->nric;
        $user->remark = $request->remark;
        $user->gender = $request->gender;
        $user->type = $request->type;
        $user->save();
        return redirect()->back()->with('success', 'User updated successfully.')->with('updated_user_id', $id);
    }

    public function update_user (Request $request){
        $request->validate([
            'phone_no' => 'nullable|digits_between:8,15', // Phone number can be null, but should be between 10 and 15 digits
            'address' => 'required|string|max:255', // Address can be null, but max 255 chars
        ]);
        $user = Auth::user();
        $user->phone_no = $request->phone_no;
        $user->address = $request->address;
        $user->save();
        return redirect()->back()->with('success', 'User updated successfully.');
    }
    
    public function update_password_by_id (Request $request, $id){
        $user = User::findOrFail($id);
        $request->validate(['password' => 'required|min:8|confirmed']);
        $user->password = Hash::make($request->password);
        $user->save();
        return redirect()->back()->with('success', 'User password updated successfully.');
    }
    
    public function update_password (Request $request){
        //Validate the current password first
        $request->validate([
            'current_password' => 'required',
            'new_password' => 'required|min:8|confirmed',        
        ]);
        $user = Auth::user();
        if (!Hash::check($request->current_password, $user->password)) {
            return redirect()->back()->withErrors(['new_password' => 'The current password is incorrect.']);
        }
        $user->password = Hash::make($request->new_password);
        $user->save();
        return redirect()->back()->with('success', 'User password updated successfully.');
    }
    
    public function disable_user_by_id (Request $request, $id){
        $user = User::findOrFail($id);
        $user->status = "Not Active";
        $user->save();
        return redirect()->back()->with('success', 'User disabled successfully.');
    }

    public function enable_user_by_id (Request $request, $id){
        $user = User::findOrFail($id);
        $user->status = "Active";
        $user->save();
        return redirect()->back()->with('success', 'User enabled successfully.');
    }
    
    public function delete_user_by_id ($id){
        $user = User::findOrFail($id);
        $user->delete();
        return redirect()->back()->with('success', 'User deleted successfully.');
    }
    
    public function update_user_image_by_id (Request $request, $id){
        $request->validate([
            'image_path' => 'nullable|image|max:2048',
        ]);

        $user = User::findOrFail($id);

        // Delete old image if a new one is being uploaded
        if ($request->file('image_path') && $user->image_path) {
            Storage::disk('public')->delete($user->image_path);
        }

        if ($request->file('image_path')) {
            $image_file = $request->file('image_path');
            $filename = 'profile_' . time() . '.jpg';
            $image_file_path = '/' . $image_file->storeAs('/user/' . strtoupper($user->id) . '/profile_picture', $filename, 'public');
            $user->image_path = $image_file_path;
        }

        // Check for images to be deleted and update accordingly
        $removeField = 'remove_profile_image';
        $imagePathField = 'image_path';

        if ($request->has($removeField) && $request->input($removeField) == '1') {
            if ($user->$imagePathField && Storage::disk('public')->exists($user->$imagePathField)) {
                Storage::disk('public')->delete($user->$imagePathField);
            }
            $user->$imagePathField = null;
        }

        $user->save();

        return redirect()->back()->with('success', 'Profile image updated successfully.');
    }
}
